package mcfall.raytracer.tests;

import mcfall.math.Matrix;
import mcfall.math.NotInvertibleException;
import junit.framework.TestCase;

//  TODO - Rewrite this test class to work with Matrix, rather than SquareMatrix, since we've done away with the
//  separate SquareMatrix class
public class SquareMatrix extends TestCase {
	/**  This matrix is the composition of two transformations;
	 *   a rotation by 90 degrees around the z axis, 
	 *   and a translation by 3,4,0
	 */
	private static double[][] matrixValues = {
		{0, -1, 0, -4},
		{1, 0, 0, 3},
		{0, 0, 1, 0},
		{0, 0, 0, 1}
	};
	
	/**
	 * This matrix is the composition of the inverses of the 
	 * transformations from above, applied in the opposite order;
	 * first a translation by -3,-4,0 and then a rotation by -90 degrees
	 * around the z axis
	 */
	private static double[][] inverseValues = {
		{0, 1, 0, -3},
		{-1, 0, 0, -4},
		{0, 0, 1, 0},
		{0, 0, 0, 1}
	};
	
	private static double[][] noninvertibleValues = {
		{1, 1},
		{1, 1}
	};
	
	private mcfall.math.SquareMatrix matrix;

	private mcfall.math.SquareMatrix nonInvertibleMatrix;
	
	public SquareMatrix() {
		matrix = new mcfall.math.SquareMatrix (4, matrixValues);
		nonInvertibleMatrix = new mcfall.math.SquareMatrix (noninvertibleValues.length, noninvertibleValues);
	}

	protected void setUp() throws Exception {
		//  Ensure that matrix has the appropriate values in it,
		//  and that it is 0 based
		matrix.setFirstColumnIndex(0);
		matrix.setFirstRowIndex(0);
		
		for (int r=0; r < matrixValues.length; r++) {
			for (int c = 0; c < matrixValues[0].length; c++) {
				matrix.setValueAt(r, c, matrixValues[r][c]);
			}
		}
	}

	protected void tearDown() throws Exception {
		super.tearDown();
	}

	/*
	 * Test method for 'mcfall.raytracer.SquareMatrix.SquareMatrix(int)'
	 * Constructs a square matrix of size 2, and ensures it has the correct
	 * properties
	 */
	public void testSquareMatrixInt() {
		mcfall.math.SquareMatrix matrix = new mcfall.math.SquareMatrix(2);
		assertEquals (1, matrix.getFirstColumnIndex());
		assertEquals (1, matrix.getFirstRowIndex());
		for (int row = 1; row <= 2; row++) {
			for (int col = 1; col <= 2; col++) {
				assertEquals (0.0, matrix.getValueAt(row, col));
			}
		}
	}

	/*
	 * Test method for 'mcfall.raytracer.SquareMatrix.SquareMatrix(int, double[][])'
	 */
	public void testSquareMatrixIntDoubleArrayArray() {
		mcfall.math.SquareMatrix matrix = new mcfall.math.SquareMatrix(4, matrixValues);
		assertEquals (1, matrix.getFirstColumnIndex());
		assertEquals (1, matrix.getFirstRowIndex());
		matrix.setFirstColumnIndex(0);
		matrix.setFirstRowIndex(0);
		for (int r = 0; r < matrixValues.length; r++) {
			for (int c = 0; c < matrixValues[0].length; c++) {
				assertEquals (matrixValues[r][c], matrix.getValueAt(r, c));
			}
		}
	}

	/*
	 * Test method for 'mcfall.raytracer.SquareMatrix.createIdentityMatrix(int)'
	 */
	public void testCreateIdentityMatrix() {
		Matrix identMatrix = mcfall.math.SquareMatrix.createIdentityMatrix(3);
		for (int row = 1; row <= 3; row++) {
			for (int col = 1; col <= 3; col++) {
				if (row != col) {
					assertEquals ("Non diagonal entry", 0.0, identMatrix.getValueAt(row, col));
				}
				else {
					assertEquals ("Diagonal entry", 1.0, identMatrix.getValueAt(row, col));
				}
			}
		}
	}

	/*
	 * Test method for 'mcfall.raytracer.SquareMatrix.invert()'
	 */
	public void testInvert() {
		try {			
			mcfall.math.Matrix inverse = matrix.invert();
			assertEquals (matrix.getFirstColumnIndex(), inverse.getFirstColumnIndex());
			assertEquals (matrix.getFirstRowIndex(), inverse.getFirstRowIndex());
			for (int r = 0; r < matrixValues.length; r++) {
				for (int c = 0; c < matrixValues[0].length; c++) {
					assertEquals (inverseValues[r][c], inverse.getValueAt(r, c));					
				}
			}
		}
		catch (NotInvertibleException invalidNotInvertible) {
			fail ("Not invertible exception thrown for invertible matrix");
		}
		
		//  Test to make sure that the identity matrix's inverse is the
		//  identity matrix
		try {
			mcfall.math.SquareMatrix identity = new mcfall.math.SquareMatrix (3); 
			mcfall.math.Matrix inverse = identity.invert();
			assertTrue (identity.equals(inverse));
		}
		catch (NotInvertibleException invalidNotInvertible) {
			fail ("Not invertible exception thrown for identity matrix");
		}
		//  Now ensure that a non invertible exception is thrown for a 
		//  matrix that can't be inverted
		try {
			mcfall.math.Matrix nonexistentInverse = nonInvertibleMatrix.invert ();
		}
		catch (NotInvertibleException good){			
			return;
		}
		
		fail ("NotInvertibleException not thrown for non-invertible matrix");
	}

	/*
	 * Test method for 'mcfall.raytracer.SquareMatrix.isInvertible()'
	 */
	public void testIsInvertible() {
		assertTrue (matrix.isInvertible());
		assertFalse(nonInvertibleMatrix.isInvertible());
	}

}
